
var util = require('util');

var FibaroClient = module.exports = function(host, user, pass) {

    var get = require('simple-get'),
        waterfall = require('async/waterfall'),
        querystring = require('querystring'),
        http = require('http');

    this.rootUrl = 'http://'+ host +'/api';
    this.username = user;
    this.password = pass;
    this.lastPoll = 0;

    this.agent = new http.Agent({ maxSockets: 3 });

    this.setCredentials = function(user, pass) {
        this.username = user;
        this.password = pass;
    }

    this.call = function(action, params, callback) {
        // If no params was passed
        if(typeof params == 'function') {
            callback = params;
            params = {};
        }

        var qs = '';

        if (params) {
            qs = '?' + querystring.stringify(params);
        }

        var reqOptions = {
            'method': 'GET',
            'url': this.rootUrl +'/'+ action + qs,
            'auth': this.username + ':' + this.password,
            'headers': {
                'X-Fibaro-Version': 2
            },
            json: true,
            agent: this.agent
        };

        get.concat(reqOptions, handleResponse(callback));
    }

    this.post = function(action, id, args, callback) {

        var reqOptions = {
            'method': 'POST',
            'url': this.rootUrl + '/devices/' + id + '/action/' + action,
            'body': {args: args},
            'auth': this.username + ':' + this.password,
            'headers': {
                'X-Fibaro-Version': 2
            },
            json: true,
            agent: this.agent
        };

        get.concat(reqOptions, handleResponse(callback));
    }

    // TODO: put shortcuts in a separate file
    var client = this;
    var shortcuts = this.api = {
        // ---------------------------------------------------------------
        // Rooms
        // ---------------------------------------------------------------
        rooms: {
            list: function(callback) {
                client.call('rooms', callback);
            }
        },

        // ---------------------------------------------------------------
        // Scenes
        // ---------------------------------------------------------------
        scenes: {
            list: function(callback) {
                client.call('scenes', callback);
            }
        },

        // ---------------------------------------------------------------
        // Devices
        // ---------------------------------------------------------------
        devices: {
            list: function(callback) {
                client.call('devices', callback);
            },
            get: function(id, callback) {
                client.call('devices', { 'id': id }, callback);
            },
            turnOn: function(id, callback) {
                client.call('callAction', { 'deviceID': id, 'name': 'turnOn' }, callback);
            },
            turnOff: function(id, callback) {
                client.call('callAction', { 'deviceID': id, 'name': 'turnOff' }, callback);
            },
            setColor: function(id, rgbw, callback) {
                client.post('setColor', id, rgbw, callback);
            },
            setBrightness: function(id, brightness, callback) {
                client.post('setBrightness', id, [brightness], callback);
            },
            toggleValue: function(id, callback) {
                var self = this;
                var newVal = null;

                waterfall([
                    function getDeviceStatus(cb) {
                        self.call(id, cb);
                    },

                    function setDeviceStatus(device, cb) {
                        if(device.properties.value == 0) {
                            newVal = 1;
                            self.turnOn(id, cb);
                        } else {
                            newVal = 0;
                            self.turnOff(id, cb);
                        }
                    }
                ], function(err) {
                    callback(err, newVal);
                });
            }
        },
        refreshStates: function(callback) {
            client.call('refreshStates', { 'last': client.lastPoll }, function(err, response) {
                if (!err && response.last) {
                    client.lastPoll = response.last;
                }
                callback(err, response);
            });
        }
    };
}

function handleResponse (callback)
{
    return function (err, res, body) {
        if (res && res.statusCode == 401) {
            callback(new AuthError('Bad username or password.'));
        } else if (res && res.statusCode != 200 && res.statusCode != 202) {
            callback(new Error('Fibaro API returned status code : ' + res.statusCode));
        } else if (err) {
            if ('code' in err && err.code == 'ECONNREFUSED') {
                callback(new Error('Fibaro not running...'));
            } else {
                callback(err);
            }
        } else if (typeof body == 'object' && 'error' in body) {
            callback(new Error(body.error));
        } else {
            callback(null, body);
        }
    }
}

FibaroClient.discover = function (callback, timeout) {

    var re = /^ACK (HC(?:2|L)-[0-9]+) ([0-9a-f:]+)$/;

    var server = require('dgram').createSocket("udp4");

    server.on('message', function (packet, rinfo) {

        var matches = re.exec(packet.toString());

        if (matches) {
            callback({
                ip: rinfo.address,
                serial: matches[1],
                mac: matches[2]
            });
        }
    });

    server.on('error', function (err) {

    });

    server.bind(44444, function () {
        var message = new Buffer("FIBARO");
        server.setBroadcast(true);
        server.send(message, 0, message.length, 44444, "255.255.255.255");
    });

    setTimeout(function () {
        server.close();
    }, timeout || 5000);
};

/* Custom error objects */
var AbstractError = function (msg, constr) {
  Error.captureStackTrace(this, constr || this)
  this.message = msg || 'Error'
}
util.inherits(AbstractError, Error);
AbstractError.prototype.name = 'Abstract Error';

var AuthError = function (msg) {
  AuthError.super_.call(this, msg, this.constructor)
}
util.inherits(AuthError, AbstractError)
AuthError.prototype.message = 'Authentification Error'
